// LunchTimeCoder.ServiceMonitor
// Copyright (c) 2007 Chris Pietschmann (http://pietschsoft.com)
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using System.Xml.Serialization;
using System.IO;
using System.Xml;

namespace LunchTimeCoder.ServiceMonitor
{
    public partial class frmMain : Form
    {
        private Timer timerCheckServices;
        private bool canShowBalloonTip = true;
        private string currentNotifyIconShown = "";

        public frmMain()
        {
            InitializeComponent();

            this.Resize += new EventHandler(frmMain_Resize);
            this.Disposed += new EventHandler(frmMain_Disposed);
        }

        void frmMain_Disposed(object sender, EventArgs e)
        {
            this.notifyIcon1.Dispose();
        }

        private void LoadGoodIcon()
        {
            if (currentNotifyIconShown != "Good")
            {
                this.notifyIcon1.Icon = new Icon(typeof(frmMain), "Resources.Good.ico");
                this.notifyIcon1.BalloonTipIcon = ToolTipIcon.Info;
                currentNotifyIconShown = "Good";
            }
        }

        private void LoadBadIcon()
        {
            if (currentNotifyIconShown != "Bad")
            {
                this.notifyIcon1.Icon = new Icon(typeof(frmMain), "Resources.Bad.ico");
                this.notifyIcon1.BalloonTipIcon = ToolTipIcon.Error;
                currentNotifyIconShown = "Bad";
            }
        }

        private void Form1_Load(object sender, EventArgs e)
        {
            LoadGoodIcon();
            this.notifyIcon1.BalloonTipTitle = "LunchTimeCoder.ServiceMonitor";
            this.notifyIcon1.BalloonTipClicked += new EventHandler(notifyIcon1_BalloonTipClicked);

            LoadSavedServicesToMonitor();

            timerCheckServices = new Timer();
            timerCheckServices.Interval = 2000;
            timerCheckServices.Tick += new EventHandler(timerCheckServices_Tick);
            timerCheckServices.Start();
        }

        private void LoadSavedServicesToMonitor()
        {
            string val = Properties.Settings.Default["ServicesToMonitor"] as string;
            if (val.Length != 0)
            {
                ServiceInfo[] col;
                col = DeserializeObject<ServiceInfo[]>(Properties.Settings.Default["ServicesToMonitor"].ToString()); // x.Deserialize(s) as ArrayList;
                for (int i = 0; i < col.Length; i++)
                {
                    lbServices.Items.Add(col[i] as ServiceInfo);
                }
            }
        }

        public void SaveServicesMonitored()
        {
            ServiceInfo[] col = new ServiceInfo[lbServices.Items.Count];
            for (int i = 0; i < lbServices.Items.Count; i++)
            {
                col[i] = lbServices.Items[i] as ServiceInfo;

            }
            Properties.Settings.Default["ServicesToMonitor"] = SerializeObject(col); // s.ToString();
            Properties.Settings.Default.Save();
        }

        public T DeserializeObject<T>(string pXmlizedString)
        {
            XmlSerializer xs = new XmlSerializer(typeof(T));
            MemoryStream memoryStream = new MemoryStream(StringToUTF8ByteArray(pXmlizedString));
            XmlTextWriter xmlTextWriter = new XmlTextWriter(memoryStream, Encoding.UTF8);
            return (T)xs.Deserialize(memoryStream);
        }

        /// <summary>
        /// http://www.dotnetjohn.com/articles.aspx?articleid=173
        /// Notes about serialization: You can only use strongly typed collections. You must convert an ArrayList to just a standard Array.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="pObject"></param>
        /// <returns></returns>
        public string SerializeObject<T>(T pObject)
        {
            String XmlizedString = null;
            MemoryStream memoryStream = new MemoryStream();
            XmlSerializer xs = new XmlSerializer(typeof(T));
            XmlTextWriter xmlTextWriter = new XmlTextWriter(memoryStream, Encoding.UTF8);
            xs.Serialize(xmlTextWriter, pObject);
            memoryStream = (MemoryStream)xmlTextWriter.BaseStream;
            XmlizedString = UTF8ByteArrayToString(memoryStream.ToArray());
            return XmlizedString;
        }

        private string UTF8ByteArrayToString(Byte[] characters)
        {
            UTF8Encoding encoding = new UTF8Encoding();
            String constructedString = encoding.GetString(characters);
            return (constructedString);
        }

        private Byte[] StringToUTF8ByteArray(string pXmlString)
        {
            UTF8Encoding encoding = new UTF8Encoding();
            Byte[] byteArray = encoding.GetBytes(pXmlString);
            return byteArray;
        }

        void notifyIcon1_BalloonTipClicked(object sender, EventArgs e)
        {
            notifyIcon1_MouseDoubleClick(sender, new MouseEventArgs(MouseButtons.None, 0, 0, 0, 0));
        }

        protected override void OnClosing(CancelEventArgs e)
        {
            // Minimize when the close button is clicked
            e.Cancel = true;
            this.WindowState = FormWindowState.Minimized;

            base.OnClosing(e);          
        }

        void timerCheckServices_Tick(object sender, EventArgs e)
        {
            ServiceInfo si;
            List<string> servicesStopped = new List<string>();
            List<string> servicesUnreachable = new List<string>();
            for (int i = 0; i < lbServices.Items.Count; i++)
            {
                si = lbServices.Items[i] as ServiceInfo;
                try
                {
                    System.ServiceProcess.ServiceController controller;

                    if (si.Computer.Length == 0)
                        controller = new System.ServiceProcess.ServiceController(si.ServiceName);
                    else
                        controller = new System.ServiceProcess.ServiceController(si.ServiceName, si.Computer);

                    if (controller.Status == System.ServiceProcess.ServiceControllerStatus.Stopped)
                    {
                        servicesStopped.Add(controller.DisplayName + (si.Computer.Length != 0 ? " on " + si.Computer : ""));
                    }
                    si.ListName = si.DisplayName + " (" + controller.Status.ToString() + ")";
                }
                catch (Exception ex)
                {
                    servicesUnreachable.Add(si.DisplayName + (si.Computer.Length != 0 ? " on " + si.Computer : ""));
                    si.ListName = si.DisplayName + " (Unreachable)";
                }
            }
            if (servicesUnreachable.Count != 0)
            {
                LoadBadIcon();
                string msg = "The following services are unreachable:\n";
                foreach (string s in servicesUnreachable)
                {
                    msg += s + "\n";
                }
                notifyIcon1.Text = "Some services are Unreachable!";
                notifyIcon1.BalloonTipText = msg;
                notifyIcon1.ShowBalloonTip(1500);
            }
            else if (servicesStopped.Count != 0)
            {
                LoadBadIcon();
                string msg = "The following services are Stopped:\n";
                foreach (string s in servicesStopped)
                {
                    msg += s + "\n";
                }
                notifyIcon1.Text = "Some services are Stopped!";
                notifyIcon1.BalloonTipText = msg;
                if (canShowBalloonTip)
                {
                    notifyIcon1.ShowBalloonTip(1500);
                    canShowBalloonTip = false;
                }
            }
            else
            {
                canShowBalloonTip = true;
                LoadGoodIcon();
                notifyIcon1.Text = "All services are running.";
                notifyIcon1.BalloonTipText = notifyIcon1.Text;
            }

            // Update display of Services
            lbServices_SelectedIndexChanged(sender, e);            
        }

        void frmMain_Resize(object sender, EventArgs e)
        {
            this.ShowInTaskbar = !(this.WindowState == FormWindowState.Minimized);
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            frmServicePicker picker = new frmServicePicker();
            if (picker.ShowDialog() == DialogResult.OK)
            {
                this.lbServices.Items.Add(picker.SelectedService);
            }
            UpdateServiceCount();
            canShowBalloonTip = true;

            SaveServicesMonitored();
        }

        private void lbServices_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (lbServices.SelectedItem != null)
            {
                ServiceInfo si = lbServices.SelectedItem as ServiceInfo;
                try
                {
                    System.ServiceProcess.ServiceController controller;

                    if (si.Computer.Length == 0)
                        controller = new System.ServiceProcess.ServiceController(si.ServiceName);
                    else
                        controller = new System.ServiceProcess.ServiceController(si.ServiceName, si.Computer);

                    lblServiceState.Text = controller.Status.ToString();
                    if (controller.Status == System.ServiceProcess.ServiceControllerStatus.Running)
                    {
                        btnStop.Enabled = true;
                        btnStart.Enabled = false;
                    }
                    else if (controller.Status == System.ServiceProcess.ServiceControllerStatus.StopPending || controller.Status == System.ServiceProcess.ServiceControllerStatus.StartPending)
                    {
                        btnStop.Enabled = false;
                        btnStart.Enabled = false;
                    }
                    else
                    {
                        btnStop.Enabled = false;
                        btnStart.Enabled = true;
                    }
                }
                catch (Exception ex)
                {
                    btnStart.Enabled = false;
                    btnStop.Enabled = false;
                }
            }
            else
            {
                btnStart.Enabled = false;
                btnStop.Enabled = false;
            }
        }

        private void btnRemove_Click(object sender, EventArgs e)
        {
            this.lbServices.Items.Remove(this.lbServices.SelectedItem);
            UpdateServiceCount();
            canShowBalloonTip = true;
        }

        private void UpdateServiceCount()
        {
            this.lblServiceCount.Text = this.lbServices.Items.Count.ToString();
        }

        private void notifyIcon1_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            this.TopMost = true;
            this.WindowState = FormWindowState.Normal;
            this.Focus();
            this.TopMost = false;
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void aboutToolStripMenuItem_Click(object sender, EventArgs e)
        {
            (new frmAbout()).ShowDialog();
        }

        private void servicesToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.WindowState = FormWindowState.Normal;
            this.Focus();
        }

        private void btnStart_Click(object sender, EventArgs e)
        {
            ServiceInfo si = lbServices.SelectedItem as ServiceInfo;
            System.ServiceProcess.ServiceController controller = new System.ServiceProcess.ServiceController(si.ServiceName);
            try
            {
                controller.Start();
            }
            catch (Exception ex)
            {
                MessageBox.Show("An Error occurred Starting service:\n" + controller.DisplayName);
            }
        }

        private void btnStop_Click(object sender, EventArgs e)
        {
            ServiceInfo si = lbServices.SelectedItem as ServiceInfo;
            System.ServiceProcess.ServiceController controller = new System.ServiceProcess.ServiceController(si.ServiceName);
            try
            {
                controller.Stop();
            }
            catch (Exception ex)
            {
                MessageBox.Show("An Error occurred Stopping service:\n" + controller.DisplayName);
            }
        }
    }
}